/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xpci16.c 
 * -----------------------------------------------------------------*/

#include <xutil.h>
#ifdef BEST_DEBUG
#define DBG_OUT(a)          BESTX_PRINTF a
#else
#define DBG_OUT(a)          (void) (0)
#endif

#include <stdio.h>
#include <assert.h>

#include <xtypedef.h>

#include <xio.h>
#include <xpci.h>
#include <timeout.h>

#include <xmailbox.h>

#ifndef MAILBOX_ONLY
#  include <xsession.h>
#  include <xerrcapi.h>
#endif /* ndef MAILBOX_ONLY */


/* slotnumber and dwHPSlotId format:    bit 15..8  - bus number,
                                        bit  7..3  - unit number (slot),
                                        bit  2..0  - function
 */


#ifdef DOSW32
#include <i86.h>
#include <conio.h>
#endif /* defined DOSW32 */

#include <dos.h>
#include <stdio.h>
#include <string.h>


static bx_errtype ReadConfigSpace(bx_int32 offset,
                                  bx_int32 size,
                                  bx_int32 DeviceNumber,
                                  bx_int32 *Value);

static bx_errtype WriteConfigSpace(bx_int32 offset,
                                   bx_int32 size,
                                   bx_int32 DeviceNumber,
                                   bx_int32 Value);


#ifdef DOSW32

#define BestXPCIDWGet(Reg_N, DevNr) \
        ((bx_int32) ReadPCIDWord((Reg_N), (DevNr)))

#define BestXPCIDWSet(Reg_N, DevNr, value) \
        WritePCIDWord((Reg_N),(DevNr),(value))

#elif !defined (WIN95) /* normal DOS */

/* GETPCIDWORD and PUTPCIDWORD are 16-bit DOS asm functions in file utilasm.asm */

long GETPCIDWORD(bx_int16 PCIoffset, bx_int16 DeviceNumber);
void PUTPCIDWORD(bx_int16 PCIoffset, bx_int16 DeviceNumber, long value);

#define BestXPCIDWGet(Reg_N, OsHandle) \
        ((bx_int32) GETPCIDWORD((bx_int16)(Reg_N), (bx_int16)(OsHandle)))

#define BestXPCIDWSet(Reg_N, OsHandle, value) \
        PUTPCIDWORD((bx_int16)(Reg_N), (bx_int16)(OsHandle), (long)(value))

#else /* WIN95 */


/* --------------------------------------------------------------------------
 * These inline asm functions are used by Win95 only to access PCI
 * -------------------------------------------------------------------------- */

#define BestXPCIDWGet(Reg_N, OsHandle) \
        GetPciDWord((bx_int16)(Reg_N), (bx_int16)(OsHandle))

static bx_int32 GetPciDWord(bx_int16 PCIoffset, bx_int16 DeviceNumber)
{
  bx_int32 result;

  _asm {
    mov     bx, DeviceNumber     ;bus#, device#, fct#
    mov     di, PCIoffset        ;offset address in conf.space

    mov     ax, 0b10ah           ;set PCI BIOS function to read config dword
    int     1ah                  ;call the BIOS fct

    mov     result, ecx          ;BIOS put it here
  }

  return result;
}


#define BestXPCIDWSet(Reg_N, OsHandle, value) \
        PutPciDWord((bx_int16)(Reg_N), (bx_int16)(OsHandle), (bx_int32)(value))

static void PutPciDWord(bx_int16 PCIoffset, bx_int16 DeviceNumber, bx_int32 DWordValue)
{
  _asm {
    mov     bx, DeviceNumber     ;bus#, device#, fct#
    mov     di, PCIoffset        ;offset address in conf.space
    mov     ecx, DWordValue

    mov     ax, 0b10dh           ;set PCI BIOS function to write config dword
    int     1ah                  ;call the BIOS fct
  }
}

#endif  /* WIN95 */


#ifdef DOSW32
#define CONFIG_ADDRESS 0xcf8
#define CONFIG_DATA 0xcfc
/***************************************************************************
The functions below are for the access to the PCI-Port in DOS.

BestXDirectRegRead
 -> BestXDirectRegPCIRead
    ->  ReadPCIDWord: calls ReadConfigSpace
        ReadPCIWord : calls ReadConfigSpace
        ReadPCIByte : calls ReadConfigSpace

BestXDirectRegWrite
 -> BestXDirectRegPCIWrite
    ->  WritePCIDWord: calls WriteConfigSpace
        WritePCIWord:  calls WriteConfigSpace
        WritePCIByte:  calls WriteConfigSpace


ReadConfigSpace:  Routine to read   1,2 or 4 bytes from ConfigSpace Register
WriteConfigSpace: Routine to write  1,2 or 4 btyes to ConfigSpace Register

***************************************************************************/

 
static bx_errtype ReadConfigSpace(bx_int32 offset,
                                  bx_int32 size,
                                  bx_int32 DeviceNumber,
                                  bx_int32 *Value)
{
/*
This function reads a value of 'size' (1,2 or 4bytes) from offset 'Reg_N' 
in the ConfigSpace by direcly accessing the host bridge IO registers CF8h and CFCh. 
Please refer to the 
PCI spec. 2.2 Chapter 3.2.2.3.2 Software Generation of Configuration Transactions.

'DeviceNumber' can be obtained using BestDevIdentifierGet.
*/
  unsigned long cAddrVal = (((DeviceNumber << 8) + offset) | 0x80000000) & 0xfffffffc;

  /* Disable interrupts.  */
  _asm { cli }

  outpd(CONFIG_ADDRESS, cAddrVal);

  switch(size)
  {
  case 1: 
    *Value = inp(CONFIG_DATA | (offset & 0x3));
    break;
  case 2: 
    *Value = inpw(CONFIG_DATA | (offset & 0x3));
    break;
  case 4: 
    *Value = inpd(CONFIG_DATA);
    break;
  default: 
    *Value = inpd(CONFIG_DATA);
  }
  
  outpd(CONFIG_ADDRESS, 0);

  /* Enable interrupts. */
  _asm { sti }

  return BX_E_OK;
} /* ReadConfigSpace */


static bx_errtype WriteConfigSpace(bx_int32 offset,
                                   bx_int32 size,
                                   bx_int32 DeviceNumber,
                                   bx_int32 Value)
{
/*
This function writes a value of 'size' (1,2 or 4bytes) to offset 'Reg_N' 
in the ConfigSpace by direcly accessing the host bridge IO registers CF8h and CFCh. 
Please refer to the 
PCI spec. 2.2 Chapter 3.2.2.3.2 Software Generation of Configuration Transactions.

'DeviceNumber' can be obtained using BestDevIdentifierGet.
*/
  unsigned long cAddrVal = (((DeviceNumber << 8) + offset) | 0x80000000) & 0xfffffffc;

  /* Disable interrupts.  */
  _asm { cli }

  outpd(CONFIG_ADDRESS, cAddrVal);
  
  switch(size)
  {
  case 1: 
    outp(CONFIG_DATA | (offset & 0x3), Value);
    break;
  case 2: 
    outpw(CONFIG_DATA | (offset & 0x3), Value);
    break;
  case 4: 
    outpd(CONFIG_DATA, Value);
    break;
  default: 
    outpd(CONFIG_DATA, Value);
  }
  
  outpd(CONFIG_ADDRESS, 0);

  /* Enable interrupts. */
  _asm { sti }

  return BX_E_OK;
  
} /* WriteConfigSpace */
#undef CONFIG_ADDRESS
#undef CONFIG_DATA


static bx_int32 ReadPCIDWord(bx_int32 Reg_N, bx_int32 DeviceNumber)
{
  bx_int32 Value;
  bx_errtype err;

  err= ReadConfigSpace(Reg_N,
                 sizeof(bx_int32),
                 DeviceNumber,
                 &Value);       
  return Value;
}

static void WritePCIDWord(bx_int32 Reg_N, bx_int32 DeviceNumber, bx_int32 Value)
{
  bx_errtype err;

  err= WriteConfigSpace(Reg_N,
                 sizeof(bx_int32),
                 DeviceNumber,
                 Value);       
}

static void WritePCIWord(bx_int32 Reg_N, bx_int32 DeviceNumber, bx_int16 Value)
{
  bx_errtype err;

  err= WriteConfigSpace(Reg_N,
                 sizeof(bx_int16),
                 DeviceNumber,
                 Value);       
}

static bx_int16 ReadPCIWord(bx_int32 Reg_N, bx_int32 DeviceNumber)
{
  bx_int32 Value;
  bx_errtype err;

  err= ReadConfigSpace(Reg_N,
                 sizeof(bx_int16),
                 DeviceNumber,
                 &Value);       
  DBG_OUT(("ReadPCIWord returned 0x%x\n",Value));
  Value=(bx_int16) Value;
  DBG_OUT(("ReadPCIWord casted in 0x%x\n",Value));
  return Value;
}

static void WritePCIByte(bx_int32 Reg_N, bx_int32 DeviceNumber, bx_int8 Value)
{
  bx_errtype err;

  err= WriteConfigSpace(Reg_N,
                 sizeof(bx_int8),
                 DeviceNumber,
                 Value);       
}

static bx_int8 ReadPCIByte(bx_int32 Reg_N, bx_int32 DeviceNumber)
{
  bx_int32 Value;
  bx_errtype err;

  err= ReadConfigSpace(Reg_N,
                 sizeof(bx_int8),
                 DeviceNumber,
                 &Value);       
  return (bx_int8) Value;
}

#ifndef MAILBOX_ONLY

bx_errtype EXPORT BestXDirectRegPCIRead(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,
  bx_int32 *reg_value
)
/*
This function uses the access_port to receive data from a certain 
memory-location on the card.
Each call to 'BestXDirectRegRead' is redirected to here.
*/

{
  bx_int32 devid;
  bx_int32 readresult;
  bx_int32 loop = 0;
  bx_int32 size;

  BX_TRY_VARS_NO_PROG;
 
  BX_TRY_BEGIN
  {
    /* bits 2::1 in access_cmd register indicate the size */
    switch(regsize)
    {
      case 1:
        size=0x2;
        break;
      case 2:
        size=0x4;
        break;
      case 4:
        size=0x0;
        break;
      default:
        size=0x0;
    }

    /* BestXDevIdentifierGet(0x15bc,0x2929,0,&devid); */
    devid=bx_handlearray[handle].portnumber;

    /* 0. Loop waiting on port_rdy bit from 'access_stat' status reg for 1 sec */
    while( !(ReadPCIDWord(ACCESS_STAT,devid) & 0x01) ) 
    {
      delay(100);
      if (++loop > 10) 
      {
        DBG_OUT(("DirectRegPCIRead: port_rdy bit not found clear after 10 loops at 100ms!\n"));
        break;
      }

    }

    /* 1. Read port_rdy bit from 'access_stat' status reg */
    if(ReadPCIDWord(ACCESS_STAT,devid) & 0x01)
    {
      /* 2./3. If port_rdy bit is set, write address and command */
      WritePCIDWord(ACCESS_CMD, devid, ( (dir_addr<<8) | 0x01 | size) );
  
      /* 4. read data_rdy bit until it returns 1*/
      while( !(ReadPCIDWord(ACCESS_STAT,devid) & 0x02) ) {;}

      /* 5. now read the value */
      switch(regsize)
      {
        case 1:
          readresult=(bx_int8)ReadPCIByte(ACCESS_DATA,devid);
          break;
        case 2:
          readresult=(bx_int16)ReadPCIWord(ACCESS_DATA,devid);
          break;
        case 4: 
          readresult=ReadPCIDWord(ACCESS_DATA,devid);
          break;
        default:
          readresult=ReadPCIDWord(ACCESS_DATA,devid);

      }
      DBG_OUT(("{Read 0x%lx from address 0x%lx}",readresult,dir_addr));
      *reg_value=readresult;
    }
    else /* the port_rdy bit was not set ! */
    {
      DBG_OUT(("DirectRegPCIRead: The port_rdy bit was not set !\n"));
      BX_E_ERROR_MSG_SET("The port_rdy bit was not set !");
      BX_ERRETURN(BX_E_ERROR);
    }

  } /*BX_TRY_END*/

  BX_ERRETURN(BX_E_OK);

} /* BestXDirectRegPCIRead */


bx_errtype EXPORT BestXDirectRegPCIWrite(
  bx_handletype handle,
  bx_int32 dir_addr,
  bx_int32 regsize,    
  bx_int32 reg_value
)
/*
This function uses the access_port to write data to a certain 
memory-location on the card.
Each call to 'BestXDirectRegWrite' is redirected to here.
*/

{
  bx_int32 devid;
  bx_int32 writevalue;
  bx_int32 size;
  bx_int32 loop = 0;
  bx_int32 i;

  BX_TRY_VARS_NO_PROG;

 
  BX_TRY_BEGIN
  {
    /* bits 2::1 in access_cmd register indicate the size */
    switch(regsize)
    {
      case 1:
        size=0x2;
        break;
      case 2:
        size=0x4;
        break;
      case 4:
        size=0x0;
        break;
      default:
        size=0x0;
    }

/*    BestXDevIdentifierGet(0x15bc,0x2929,0,&devid); */
    devid=bx_handlearray[handle].portnumber;

    /* 0. Loop waiting on port_rdy bit from 'access_stat' status reg for 1 sec */
    while( !(ReadPCIDWord(ACCESS_STAT,devid) & 0x01) ) 
    {
      delay(100);
      if (++loop > 10) 
      {
        DBG_OUT(("DirectRegPCIWrite: port_rdy bit not found clear after 10 loops at 100ms!\n"));
        break;
      }
    }

    /* 1. Read port_rdy bit from 'access_stat' status reg */
    if(ReadPCIDWord(ACCESS_STAT,devid) & 0x01)
    {
      /* 2./3. If port_rdy bit is set, write address and command */
      WritePCIDWord(ACCESS_CMD, devid, ( (dir_addr<<8) | 0x00 | size) );

      /* 4. now write in data-register  */
     
      switch(regsize)
      {
        case 1:
          WritePCIByte(ACCESS_DATA,devid,(bx_int8)reg_value);
          break;
        case 2:
          WritePCIWord(ACCESS_DATA,devid,(bx_int16)reg_value);
          break;
        case 4: 
          WritePCIDWord(ACCESS_DATA,devid,reg_value);
          break;
        default:
          WritePCIDWord(ACCESS_DATA,devid,reg_value);

      }
     DBG_OUT(("{Write 0x%lx to address 0x%lx}",reg_value,dir_addr));
    }
    else /* the port_rdy bit was not set ! */
    {
      DBG_OUT(("DirectRegPCIWrite: The port_rdy bit was not set !\n"));
      BX_E_ERROR_MSG_SET("The port_rdy bit was not set !");
      BX_ERRETURN(BX_E_ERROR);
    }

  } /*BX_TRY_END*/

  BX_ERRETURN(BX_E_OK);
} /* DirectRegPCIWrite */

#endif /* not defined MAIlBOX_ONLY */

#endif /* defined DOSW32 */
/**********************************************************************/


/* --------------------------------------------------------------------------
 * BestXPCISetRegwidth() for DOS uses the handle array
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCISetRegwidth(bx_portnumtype OsHandle, int regwidth)
{
  BX_USE(OsHandle);
  BX_USE(regwidth);
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXDeviceConnect()
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCIDeviceConnect(bx_portnumtype OsHandle)
{
  BestXPCIDWSet(PCI_CONNECT_CMD, OsHandle, PCI_CONNECT_CMD_BIT);
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXCheckConnection()
 * -------------------------------------------------------------------------- */

bx_errtype BestXPCICheckConnection(bx_portnumtype OsHandle)
{
  if ((BestXPCIDWGet(PCI_CONNECT_STATUS, OsHandle) & PCI_CONNECT_STATUS_BIT) == 0)
    return BX_E_NOT_CONNECTED;
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXReleaseConnection()
 * -------------------------------------------------------------------------- */

void BestXPCIReleaseConnection(bx_portnumtype OsHandle)
{
  BestXPCIDWSet(PCI_CONNECT_CMD, OsHandle, 0x0);
}


/* --------------------------------------------------------------------------
 * Should only be called from BestXPortTimeoutSet()
 * Handle array is used to store timeouts.
 * -------------------------------------------------------------------------- */

bx_errtype BestXPciPortTimeoutSet(bx_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
  BX_USE(OsHandle);
  BX_USE(pCallersTimeouts);
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Because not all single-function devices correctly respond to a read
 * of functions 1 through 7 (even though they're not implemented) this
 * function allows a direct way of determining the type of device.
 * -------------------------------------------------------------------------- */

bx_errtype BestXPciDevMultiFuncCheck(bx_int32 dwHPSlotId, bx_bool * fIsMultiFunc)
{
  *fIsMultiFunc = 0;

  if ( (BestXPCIDWGet(0, dwHPSlotId) & 0xffff) == 0xffff )
    return BX_E_NO_BEST_PCI_DEVICE_FOUND;

  *fIsMultiFunc = ((BestXPCIDWGet(PCI_MF_OFFSET, dwHPSlotId) & PCI_MF_BIT) != 0);
  return BX_E_OK;
}


#ifndef MAILBOX_ONLY

/* --------------------------------------------------------------------------
 * Does an 8, 16 or 32-bit series of READS from the PCI programming port.
 * NOTE; This is NOT the same as byte-ordered (!!!).
 * -------------------------------------------------------------------------- */

bx_errtype BestXPciBasicRead(
    bx_handletype handle,
    bx_int8ptr pData,
    bx_int32 ulBytesToRead,
    bx_int32 * ulBytesRead)
{
  /* the OsHandle is the port offset */
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;
  bx_int8 regwidth = bx_handlearray[handle].regwidth;

  BEST_TIMER  tmrInterval, tmrTotal;  /* SCR; 2.12.97 */

  /* protect against bad stuff at run-time. */
  if ( (0 == ulBytesToRead) || (NULL == pData) )
  {
    return BX_E_PARAM_OUT_OF_RANGE;
  }

  /* SCR; 27.11.97 init timers */
  BestInitReadTimeouts(&tmrInterval, &tmrTotal, 
                       &(bx_handlearray[handle].timeouts), ulBytesToRead);

  *ulBytesRead = 0;

  while(*ulBytesRead < ulBytesToRead)
  {
    /* wait for valid data */
    if ((BestXPCIDWGet(PCI_STATUS, OsHandle) & PCI_STATUS_DATA_VALID_BIT) != 0)
    {
      switch ( regwidth )
      {
      case 1:
        *pData = (bx_int8) BestXPCIDWGet(PCI_DATA, OsHandle);
        break;

      case 2:
        *((bx_int16 *)pData) = (bx_int16) BestXPCIDWGet(PCI_DATA, OsHandle);
        break;

      case 4:
        *((bx_int32 *)pData) = BestXPCIDWGet(PCI_DATA, OsHandle);
        break;

      default:
        assert(0 && "Illegal reg. width in BestXPCIBasicRead()");
        return BX_E_PARAM_OUT_OF_RANGE;
      }
      DBG_OUT(("[Read 0x%lx]",*pData));

      pData += regwidth;
      *ulBytesRead += regwidth;
      BestXPCIDWSet(PCI_STATUS, OsHandle, PCI_STATUS_DATA_VALID_BIT);
      BestRestartIntervalTimeout(&tmrInterval); /* MUST do */
    }
    else 
    {
      if ( BX_E_OK != BestXPCICheckConnection(OsHandle) )
        return BX_E_NOT_CONNECTED;

      if (BestIsReadTimeoutDone(&tmrInterval, &tmrTotal))
        break;  /* not an error */
    }
  }

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Does an 8, 16 or 32-bit series of WRITES to the PCI port.
 * NOTE; This is NOT the same as byte-ordered (!!!).
 * -------------------------------------------------------------------------- */

bx_errtype BestXPciBasicWrite(
    bx_handletype handle,
    bx_int8ptr pData,
    bx_int32 ulBytesToWrite,
    bx_int32 * ulBytesWritten)
{
  /* the OsHandle is the port offset */
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;
  bx_int8 regwidth = bx_handlearray[handle].regwidth;

  BEST_TIMER         tmrTotal;    /* SCR; 2.12.97 */

  /* protect against bad stuff at run-time. */
  if ( (0 == ulBytesToWrite) || (NULL == pData) )
  {
    return BX_E_PARAM_OUT_OF_RANGE;
  }
  /* SCR; 2.12.97 init timer using handle timeouts */
  BestInitWriteTimeout(&tmrTotal, &(bx_handlearray[handle].timeouts), ulBytesToWrite);

  *ulBytesWritten = 0;

  /* write data ... */
  while(*ulBytesWritten < ulBytesToWrite)
  {
    /* wait for send register empty */
    if ((BestXPCIDWGet(PCI_STATUS, OsHandle) & PCI_STATUS_SEND_FULL_BIT) == 0)
    {
      DBG_OUT(("[Write 0x%lx]",*pData));
      switch ( regwidth )
      {
      case 1:
        BestXPCIDWSet(PCI_DATA, OsHandle, (*pData));
        break;

      case 2:
        BestXPCIDWSet(PCI_DATA, OsHandle, (*((bx_int16 *)pData)));
        break;

      case 4:
        BestXPCIDWSet(PCI_DATA, OsHandle, (*((bx_int32 *)pData)));
        break;

      default:
        assert(0 && "Illegal reg. width in BestXPCIBasicWrite()");
        return BX_E_PARAM_OUT_OF_RANGE;
      }
    
      pData += regwidth;
      *ulBytesWritten += regwidth;
    }
    else 
    {
      if ( BX_E_OK != BestXPCICheckConnection(OsHandle) )
        return BX_E_NOT_CONNECTED;

      if ( BestIsWriteTimeoutDone(&tmrTotal) )
        break;  /* a timeout is NOT an error */
    }
  }

  return BX_E_OK;
}

#endif /* defined MAILBOX_ONLY */

/******************************************************************************/
bx_errtype BestXGenericPCIDWSet(bx_int32 dwHPSlotId,
    bx_int32 Reg_N,
    bx_int32 value)
{
  BestXPCIDWSet(Reg_N, dwHPSlotId, value);
  return BX_E_OK;
}


/******************************************************************************/
bx_errtype BestXGenericPCIDWGet(bx_int32 dwHPSlotId,
    bx_int32 Reg_N,
    bx_int32 * value)
{
#if 0
  printf("We are in BestXGenericPCIDWGet. Just return BX_E_OK.. \n");
#endif
  *value = BestXPCIDWGet(Reg_N, dwHPSlotId);
  return BX_E_OK;
}


/******************************************************************************/
bx_errtype BestXOpenPCI(int intHPSlotId, bx_portnumtype * pOsHandle)
{
  *pOsHandle = (bx_portnumtype) intHPSlotId;
  return BX_E_OK;
}


/******************************************************************************/
bx_errtype BestXOpenIO(int intHPSlotId, bx_portnumtype * pOsHandle)
{
  DBG_OUT(("We are in BestXOpenPCI. The portnr is 0x%lx\n",*pOsHandle));
  return BestXOpenPCI(intHPSlotId, pOsHandle);
}


/******************************************************************************/
bx_errtype BestXClosePCI(bx_portnumtype OsHandle)
{
  BestXPCIReleaseConnection(OsHandle);      /* just in case */
  return BX_E_OK;
}


/******************************************************************************/
bx_errtype BestXPCIMailboxWrite(bx_int32 dwHPSlotId,
                              bx_int32 value)
{
  bx_portnumtype OsHandle = (bx_portnumtype)dwHPSlotId;
  BEST_TIMER tmr;    /* SCR; 2.12.97 */

  /* SCR; 3.12.97 init timer using mailbox timeouts */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_WRITE, &tmr);

  while (0 != (BestXPCIDWGet(PCI_MBOX_STATUS, OsHandle) & PCI_MBOX_SEND_FULL_BIT))
  {
    if ( BestIsTimeoutDone(&tmr) ) {
       return BX_E_MAILBOX_TIMEOUT;  /* a timeout here is an error */
    }
  }

  /* write data to Config */
  BestXPCIDWSet(PCI_MBOX_DATA, OsHandle, value);
  return BX_E_OK;
}


/******************************************************************************/
bx_errtype BestXPCIMailboxRead(bx_int32 dwHPSlotId,
                             bx_int32 * pValue)
{
  bx_portnumtype OsHandle = (bx_portnumtype)dwHPSlotId;
  BEST_TIMER  tmr;  /* SCR; 3.12.97 */

  /* protect against bad stuff at run-time. */
  if ( NULL == pValue )
  {
    return BX_E_PARAM_OUT_OF_RANGE;
  }

  /* SCR; 3.12.97 init timeout */
  BestStartTimeout(TIMEOUT_MAILBOX_PCI_READ, &tmr);

  while (0 == (BestXPCIDWGet(PCI_MBOX_STATUS, OsHandle) & PCI_MBOX_DATA_VALID_BIT))
  {
    if ( BestIsTimeoutDone(&tmr) ) {
       return BX_E_MAILBOX_TIMEOUT;               /* a timeout here is an error */
    }
  }

  /* get data from data register */
  *pValue = BestXPCIDWGet(PCI_MBOX_DATA, OsHandle);

  /* clear the data valid bit */
  BestXPCIDWSet(PCI_MBOX_STATUS, OsHandle, PCI_MBOX_DATA_VALID_BIT);

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * These 2 functions are here to allow Win95 and 16-bit compilers
 * direct I/O access when using different compilers.
 * Borland will not allow inport/outport when compiling with _WIN32.
 * Neither of these functions make any assumptions about calling methods.
 * DIRECT_IO is defined in bx_io.h
 * -------------------------------------------------------------------------- */

#if(DIRECT_IO)

void BestXOutPortB(bx_int16 portid, bx_int8 uchar)
{
#ifndef DOSW32 
    _asm {
    xor     ax, ax
    mov     dx, portid
    mov     al, uchar
    out     dx, al
  }
#endif
}

bx_int8 BestXInPortB(bx_int16 portid)
{
bx_int8 uchar=0;
#ifndef DOSW32

  _asm {
    mov     dx, portid
    in      al, dx
    mov     uchar, al
  }
#endif
  return uchar;

}

#endif  /* DIRECT_IO */
